﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using DicomObjects.DicomCodecs;
using DicomObjects.Enums;
using DicomObjects;
namespace CodecExample
{
    class MyCodecFactory : CodecFactory
    {
        static string TransferSyntax = "1.2.3.4.5";

        public override bool CheckStart(byte[] data, string TransferSyntax)
        {
            return true; // we don't have specific markers for this type
        }

        public override ICompressor Compressor(string TransferSyntax)
        {
            return new MyCompressor();
        }

        public override IDecompressor Decompressor(string TransferSyntax, System.IO.Stream Data, DicomObjects.DicomDataSet DataSet, int Frame)
        {
            return new MyDecompressor();
        }

        public override IExporter Exporter(string Format)
        {
            throw new NotImplementedException();
        }

        public override IImporter Importer(string Format)
        {
            throw new NotImplementedException();
        }

        public override string[] TransferSyntaxes(CodecOperation Operation)
        {
            switch (Operation)
            {
                case CodecOperation.Compress:
                case CodecOperation.Decompress:
                    return new string[] { TransferSyntax };

                default:
                    return new string[0];
            }
        }
    }


    class MyCompressor : ICompressor
    {
        public void Compress(CompressionArguments args)
        {
            if (args.SourceArray is byte[])
                MainCompress(args.SourceArray as byte[], args);
            else
            {
                MainCompress(args.SourceArray as ushort[], args);
            }

        }

        private void MainCompress(byte[] array, CompressionArguments args) 
        {
            byte factor = 2;
            if(args.Quality is byte) 
                factor = (byte)args.Quality;
            if (args.Quality is int)
                factor = (byte)(int)args.Quality;

            args.DestinationStream.WriteByte(factor);

            for (int y = 0; y < args.Size.Height ; y+=factor)
            {
                for (int x = 0; x < args.Size.Width; x += factor)
                {
                    args.DestinationStream.WriteByte(array[y * args.Size.Width + x]);
                }
            }
        }
        private void MainCompress(ushort[] array, CompressionArguments args)
        {
            byte factor = 2;
            if (args.Quality is byte)
                factor = (byte)args.Quality;
            if (args.Quality is int)
                factor = (byte)(int)args.Quality;

            args.DestinationStream.WriteByte(factor);

            for (int y = 0; y < args.Size.Height; y += factor)
            {
                for (int x = 0; x < args.Size.Width; x += factor)
                {
                    byte[] pixel = BitConverter.GetBytes(array[y * args.Size.Width + x]);
                    args.DestinationStream.Write(pixel, 0, 2);
                }
            }
        }
        public int MaximumBits
        {
            get { return 16; }
        }

        public bool NeedsPlanarConfigTransform
        {
            get { return false; }
        }

        public void PrepareFunction(ModificationArguments ModificationArguments)
        {
            if ((ushort)ModificationArguments.DataSet[0x0028, 0x0002].Value > 1)
                throw new Exception("This only works for monochrome images");
        }

        public object LockObject()
        {
            return this;
        }
    }
    class MyDecompressor : IDecompressor
    {
        public DicomObjects.PixelRequest BestRequest(float Zoom, RequestType Sync, int Frame)
        {
            return null;
        }
        public void Decompress(DecompressionArguments args)
        {
            if (args.DestinationArray is byte[])
                MainDecompress(args.DestinationArray as byte[], args);
            else
            {
                MainDecompress(args.DestinationArray as ushort[], args);
            }

        }
        internal void MainDecompress(byte[] array, DecompressionArguments args)
        {
            int factor = args.SourceStream.ReadByte();

            for (int y = 0; y < args.Size.Height ; y+=factor)
            {
                for (int x = 0; x < args.Size.Width; x += factor)
                {
                    byte val = (byte)args.SourceStream.ReadByte();
                    for(int y1=0;y1<factor;y1++)
                        for (int x1 = 0; x1 < factor; x1++)
                        {
                            if(y+y1 < args.Size.Height && x+x1 < args.Size.Width)
                                array[(y + y1) * args.Size.Width + x + x1] = val;
                        }
                }
            }
        }
        internal void MainDecompress(ushort[] array, DecompressionArguments args)
        {
            int factor = args.SourceStream.ReadByte();
            byte[] valbytes = new byte[2];

            for (int y = 0; y < args.Size.Height; y += factor)
            {
                for (int x = 0; x < args.Size.Width; x += factor)
                {
                    args.SourceStream.Read(valbytes,0,2);
                    ushort val = BitConverter.ToUInt16(valbytes,0);

                    for (int y1 = 0; y1 < factor; y1++)
                        for (int x1 = 0; x1 < factor; x1++)
                        {
                            if (y + y1 < args.Size.Height && x + x1 < args.Size.Width)
                                array[(y + y1) * args.Size.Width + x + x1] = val;
                        }
                }
            }
        }
        public DecompressStatus DecompressState(DicomObjects.PixelRequest Request)
        {
            return DecompressStatus.Complete;
        }

        public string DecompressedPhotmetricInterpretation(DicomObjects.DicomDataSet parent)
        {
            return parent[0x0028, 0x0004].ToString();
        }

        public ProgressiveStatus DownloadStatus(DicomObjects.PixelRequest Request)
        {
            return ProgressiveStatus.WholeImage;
        }

        public bool NeedsPlanarConfigTransform
        {
            get { return false; }
        }

        public object LockObject()
        {
            return this;
        }
    }

}
